#!/bin/bash

# Copyright 2008-2009 Junior (Frederic) FLEURIAL MONFILS
#
# This file is part of Qualoss-Tools
#
# Qualoss-Tools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# or see <http://www.opensource.org/licenses/gpl-3.0.html>
#
# Contact:
#     Junior FLEURIAL MONFILS <frederic dot fleurialmonfils at cetic dot be>
#     Jean-Christophe DEPREZ <jean-christophe dot deprez at cetic dot be>

source config.sh

usage() {
  cat <<EOT
Usage::
$0 [-e] [-u] -a ARCHIVE -l LANGUAGE -n NAME v1 [v2 ...]

Extract source code for the given language from the source and versions
that are provided

Example::
$0 -e -u -a tar.gz -l cpp -n httpd13 1.3.9 1.3.22 1.3.29

Options::
    -h
    -a ARCHIVE    type or archive (ex: tar.gz, zip, tgz, tar.bz2, etc)
    -l LANGUAGE   language (java, cpp, python) [default: cpp]
    -e            extract only [default: False]
    -u            unzip only [default: False]
    -n NAME       name of the project
EOT
}

aEXTRACT=0
aUNZIP=0

while getopts "l:a:n:euh" flag; do
  case "$flag" in
    l) aLANG=$OPTARG;;
    a) aARCH=$OPTARG;;
    n) aNAME=$OPTARG;;
    e) aEXTRACT=1;;
    u) aUNZIP=1;;
    h) usage; source clean.sh; exit 2;;
    \?) usage; echo "Error: Invalid option"; source clean.sh; exit 3;; 
  esac
done

if [ "${aLANG}" == "cpp" ]; then
  vMATCH=".*\.(c|cc|cpp|h|hh|hpp)"
elif [ "${aLANG}" == "java" ]; then
  vMATCH=".*.java"
elif [ "$aLANG" == "python" ]; then
  vMATCH=".*\.py"
fi

if [ "${aARCH}" == "tar.gz" ]; then
  vUNZIP=tar
  vOPT=xvf
elif [ "${aARCH}" == "zip" ]; then
  vUNZIP=unzip
  vOPT=-qC
fi

curdir=`pwd`
cd ${gPROJECTS_DIR}

shift $((OPTIND-1))
declare -a versions
numargs=$#
for version; do
  if [ "$aUNZIP" == "1" ]; then
    echo unzipping archive/${aNAME}-${version}.$aARCH
    $vUNZIP $vOPT archive/${aNAME}-${version}.$aARCH
  fi
  if [ "$aEXTRACT" == "1" ]; then
    echo extracting sources to ${PWD}/sources/${aNAME}-${version}
    (find ${aNAME}-${version}/ -regextype posix-extended -iregex ${vMATCH} | xargs tar cf - ) | (cd sources && tar xpf -)
  fi
done
cd $curdir

echo done.

source clean.sh
