#!/bin/bash

# Copyright 2008-2009 Junior (Frederic) FLEURIAL MONFILS
#
# This file is part of Qualoss-Tools
#
# Qualoss-Tools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# or see <http://www.opensource.org/licenses/gpl-3.0.html>
#
# Contact:
#     Junior FLEURIAL MONFILS <frederic dot fleurialmonfils at cetic dot be>
#     Jean-Christophe DEPREZ <jean-christophe dot deprez at cetic dot be>

source config.sh

usage() {
  cat <<EOT
Usage::
$0 -l LANGUAGE -n NAME v

Compute the maintainability measures for the code of the current version

Example::
$0 -l cpp -n httpd13 1.3.41

Options::
    -h
    -l LANGUAGE   language (java, cpp, python)
    -n NAME       name of the assessment
EOT
}

while getopts "l:n:h" flag; do
  case "$flag" in
    l) aLANG=$OPTARG;;
    n) aNAME=$OPTARG;;
    h) usage; source clean.sh; exit 2;;
    \?) usage; echo "Error: Invalid option" >&2; source clean.sh; exit 3;; 
  esac
done

if [ "$aLANG" == "cpp" ]; then
  vMATCH=".*\.(cpp|cc|c|hpp|h)$"
  vSLOCMATCH='s/^(cpp|ansic):[ \t]+([0-9]+).*/\2/p'
elif [ "$aLANG" == "java" ]; then
  vMATCH=".*\.java$"
  vSLOCMATCH='s/^java:[ \t]+([0-9]+).*/\1/p'
elif [ "$aLANG" == "python" ]; then
  vMATCH=".*\.py$"
fi 

vCURDIR=`pwd`
vASSESSMENT_DIR=${gASSESSMENT_DIR}/${aNAME}
cd ${vASSESSMENT_DIR}

shift $((OPTIND-1))
v=$1

#
# Analysis of the Imports/includes
#

echo analysing imports/includes for cyclic dependencies...

if [ "$aLANG" == "java" ]; then
  vEXCLUDE="^java"
elif [ "$aLANG" == "cpp" ]; then
  # http://www.utas.edu.au/infosys/info/documentation/C/CStdLib.html
  # http://en.wikipedia.org/wiki/C_standard_library
  vEXCLUDE="<?(assert|complex|ctype|errno|fenv|float|inttypes|iso646|limits|locale|math|setjmp|signal|stdarg|stdbool|stdint|stddef|stdio|stdlib|string|tgmath|time|wchar|wctype)(\.h)?>?"
elif [ "$aLANG" == "python" ]; then
,,vEXCLUDE="^(os|sys|math)(\.?|$)"
fi

python ${gSCRIPTS_DIR}/pydepend.py -l $aLANG --internal -e "$vEXCLUDE" --dot=module_dependencies_$v.dot --csv=module_couplings_$v.csv < filelist_$v.txt 
python ${gSCRIPTS_DIR}/pydepend.py -l $aLANG --internal --files -e "$vEXCLUDE" --dot=file_dependencies_$v.dot --csv=file_couplings_$v.csv < filelist_$v.txt 

cd $vASSESSMENT_DIR
python ${gSCRIPTS_DIR}/findcycles.py -c -a -q < module_dependencies_$v.dot > module_cycles_$v.txt
#python ${gSCRIPTS_DIR}/findcycles.py -c -a -q < file_dependencies_$v.dot > file_cycles_$v.txt

#
# Number of high level modules
#

res=$(cat module_couplings_$v.csv | wc -l)
let res--
echo Number_of_high_level_modules = $res

#
# Efferent coupling of high level modules
#

res=`awk -f $gSCRIPTS_DIR/sum_efferent_coupling.awk module_couplings_$v.csv`
echo Efferent_coupling_of_high_level_modules = $res

#
# Number of high-level modules involved in dependency cycles
#

res=`sed -rn 's/Nodes involved in cycles: ([0-9]+)/\1/p' module_cycles_$v.txt`
echo Number_of_high_level_modules_involved_in_dependency_cycles = $res

#
# Number of low level modules
#

res=$(cat file_couplings_$v.csv | wc -l)
# remove 1 from the header line
let res--
echo Number_of_low_level_modules = $res

#
# Efferent coupling of low level modules
#

res=`awk -f $gSCRIPTS_DIR/sum_efferent_coupling.awk file_couplings_$v.csv`
echo Efferent_coupling_of_low_level_modules = $res

#
# Number of defined routines
#

res=`cat api_${v}.txt | wc -l`
echo Number_of_defined_routines = $res

#
# Cyclomatic complexity of defined routines
#

if [ "${aLANG}" == "java" ]; then
  #num_routines=`cat api_${v}.txt | wc -l`
  #avg_cyclo=`${gSCRIPTS_DIR}/xmlfilter.py -n -f javancss_${v}.xml -c '/javancss/functions/function_averages/ccn/text()'`
  #res=`echo "$num_routines * $avg_cyclo" | bc`
  res=`awk -f $gSCRIPTS_DIR/sum_cyclomatic_complexity_java.awk javacccs_$v.csv`
elif [ "${aLANG}" == "cpp" ]; then
  res=`tail -1 pmccabe_${v}.txt | sed -r 's/[0-9]+\s+([0-9]+).*/\1/'`
elif [ "$aLANG" == "python" ]; then
  res=`awk -f $gSCRIPTS_DIR/sum_cyclomatic_complexity_python.awk pyanalyzer_$v/project_metrics.csv`
fi
echo Cyclomatic_complexity_of_defined_routines = $res

#
# Number of lines of comments in defined routines
#

#if [ "${aLANG}" == "java" ]; then
#  awk -f ${gSCRIPTS_DIR}/select_10_random_files.awk filelist_${v}.txt > randomfiles_${v}.txt
#  sum_header_lines_in_10_random_files=`xargs -a randomfiles_${v}.txt awk -f ${gSCRIPTS_DIR}/show_licence_header.awk | wc -l`
#  avg_header_lines_per_file=`echo "0.1 * $sum_header_lines_in_10_random_files" | bc`
#  num_files=`cat filelist_${v}.txt | wc -l`
#  number_of_licence_header_lines=`echo "$avg_header_lines_per_file * $num_files" | bc`
#fi

if [ "${aLANG}" == "java" ]; then
  #sum_single_comments=`${gSCRIPTS_DIR}/xmlfilter.py -n -x '/javancss/packages/total/single_comment_lines/text()' -f javacount_${v}.xml`
  #sum_multi_comments=`${gSCRIPTS_DIR}/xmlfilter.py -n -x '/javancss/packages/total/multi_comment_lines/text()' -f javacount_${v}.xml`
  #res=`echo "$sum_single_comments + $sum_multi_comments - $number_of_licence_header_lines" | bc`
  res=$(awk -f $gSCRIPTS_DIR/sum_lines_of_comments_java.awk javacount_$v.csv)
elif [ "${aLANG}" == "cpp" ]; then
  xargs -a filelist_${v}.txt ${gTOOLS_DIR}/pmccabe/pmccabe -v -t -n > pmccabe_${v}_comments.txt
  res=$(tail -1 pmccabe_${v}_comments.txt | awk '{print $1}')
elif [ "$aLANG" == "python" ]; then
  res=$(awk -f $gSCRIPTS_DIR/sum_lines_of_comments_python.awk pyanalyzer_$v/project_metrics.csv)
fi
echo Number_of_lines_of_comments_in_defined_routines = $res

#
# Number of lines of code in defined routines
#

if [ "${aLANG}" == "java" ]; then
  #num_routines=`cat api_${v}.txt | wc -l`
  #avg_loc_per_routine=`${gSCRIPTS_DIR}/xmlfilter.py -n -f javacount_${v}.xml -x '/javancss/functions/function_averages/ncss/text()'`
  #res=`echo "$num_routines * $avg_loc_per_routine" | bc`
  res=$(awk -f $gSCRIPTS_DIR/sum_lines_of_code_java.awk javacount_$v.csv)
elif [ "${aLANG}" == "cpp" ]; then
  res=`tail -1 pmccabe_${v}.txt | awk '{print $3}'`
elif [ "$aLANG" == "python" ]; then
  res=$(awk -f $gSCRIPTS_DIR/sum_lines_of_code_python.awk pyanalyzer_$v/project_metrics.csv)
fi
echo Number_of_lines_of_code_in_defined_routines = $res

#
# End
#

cd $vCURDIR

source clean.sh
