#!/bin/bash

# Copyright 2008-2009 Junior (Frederic) FLEURIAL MONFILS
#
# This file is part of Qualoss-Tools
#
# Qualoss-Tools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# or see <http://www.opensource.org/licenses/gpl-3.0.html>
#
# Contact:
#     Junior FLEURIAL MONFILS <frederic dot fleurialmonfils at cetic dot be>
#     Jean-Christophe DEPREZ <jean-christophe dot deprez at cetic dot be>

source config.sh

usage() {
  cat <<EOT
Usage::
$0 -l LANGUAGE -n NAME v1 [v2 ...]

Do a pairwise comparison of LOC between versions of the source code

Example::
$0 -l cpp -n httpd13 1.3.9 1.3.22 1.3.29

Options::
    -h
    -l LANGUAGE   language (java, cpp, python)
    -n NAME       name of the assessment
EOT
}

while getopts "l:n:h" flag; do
  case "$flag" in
    l) aLANG=$OPTARG;;
    n) aNAME=$OPTARG;;
    h) usage; source clean.sh; exit 2;;
    \?) usage; echo "Error: Invalid option" >&2; source clean.sh; exit 3;; 
  esac
done

if [ "${aLANG}" == "cpp" ]; then
  vMATCH=".*\.(cpp|cc|c|hpp|h)$"
elif [ "${aLANG}" == "java" ]; then
  vMATCH=".*\.java$"
elif [ "${aLANG}" == "python" ]; then
  vMATCH=".*\.py$"
fi 

vCURDIR=`pwd`
vASSESSMENT_DIR=${gASSESSMENT_DIR}/${aNAME}
cd ${vASSESSMENT_DIR}

#
# Create the list of versions for pairwise comparison
#

shift $((OPTIND-1))
declare -a versions
numargs=$#
i=0
for version; do
  versions[$i]=${version}
  let i++
done

#
# Average percentage of changes in code between releases
#

echo "Step 1/4: Average_percentage_of_changes_in_code_between_releases"
echo ================================================================

i=0
for (( j=1; j<$numargs; j++ )); do
  vi=${versions[$i]}
  vj=${versions[$j]}
  ${gSCRIPTS_DIR}/locdiff.sh ${vMATCH} ${gSOURCES_DIR}/${aNAME}-${vi} ${gSOURCES_DIR}/${aNAME}-${vj} > locdiff_${vi}_${vj}.txt
  res=`tail -1 locdiff_${vi}_${vj}.txt | awk '{print $2}'`
  echo Percentage_of_changes_in_code_between_${vi}_and_${vj} = $res
  let i++
done

echo

#
# Average lines of code in successive releases
#

OLDPATH=$PATH
PATH=$PATH:${gTOOLS_DIR}/sloccount/

echo "Step 2/4: Average_lines_of_code_in_successive_releases"
echo ======================================================

for (( i=0; i<$numargs; i++ )); do
  vi=${versions[$i]}
  sloccount ${gSOURCES_DIR}/${aNAME}-${vi} > sloccount_${vi}.txt
  if [ "$aLANG" == "java" ]; then
    res=`sed -nr 's/^java:[ \t]+([0-9]+).*/\1/p' sloccount_${vi}.txt`
  elif [ "$aLANG" == "cpp" ]; then
    res=`sed -nr 's/^(cpp|ansic):[ \t]+([0-9]+).*/\2/p' sloccount_${vi}.txt | awk '{ sum += $1 }; END { print sum }'`
  elif [ "$aLANG" == "python" ]; then
    res=`sed -nr 's/^python:[ \t]+([0-9]+).*/\1/p' sloccount_${vi}.txt`
  fi
  echo Lines_of_code_in_${vi} = $res
done

PATH=$OLDPATH

echo

#
# Average changes in public interfaces between releases
#

for (( i=0; i<$numargs; i++ )); do
  vi=${versions[$i]}
  echo extracting the API of version ${vi} ...
  cd ${gSOURCES_DIR}/${aNAME}-${vi}
  find . -type f > ${vASSESSMENT_DIR}/filelist_${vi}.txt
  if [ "${aLANG}" == "java" ]; then
    #${gTOOLS_DIR}/javancss/bin/javancss -xml -out javancss_${vi}.xml -all @filelist_${vi}.txt
    #${gSCRIPTS_DIR}/xmlfilter.py -n -f javancss_${vi}.xml -x "/javancss/functions/function/name" | sort > api_${vi}.txt
    $gTOOLS_DIR/javacount/bin/javacount --java5 < ${vASSESSMENT_DIR}/filelist_${vi}.txt > $vASSESSMENT_DIR/javacount_$vi.csv
    cd $vASSESSMENT_DIR
    awk -f $gSCRIPTS_DIR/show_api_java.awk javacount_$vi.csv | sort > api_$vi.txt
  elif [ "${aLANG}" == "cpp" ]; then
    cat ${vASSESSMENT_DIR}/filelist_${vi}.txt | xargs ${gTOOLS_DIR}/pmccabe/pmccabe -v -t > $vASSESSMENT_DIR/pmccabe_${vi}.txt 2> $vASSESSMENT_DIR/pmccabe_err_${vi}.txt
    cd $vASSESSMENT_DIR
    sed -rn "s/.*[0-9]+\s+(.*)(\([0-9]+\)):\s+(.*)$/\1: \3/p" pmccabe_${vi}.txt | sort > api_${vi}.txt
  elif [ "$aLANG" == "python" ]; then
    python ${gTOOLS_DIR}/pyanalyzer/pyanalyzer.py -f $vASSESSMENT_DIR/filelist_${vi}.txt --metrics=CyclomaticComplexity,NumberOfLinesOfCode,NumberOfLinesOfComments -o $vASSESSMENT_DIR/pyanalyzer_${vi} -m -Mcsv
    cd $vASSESSMENT_DIR
    awk -f ${gSCRIPTS_DIR}/show_api_python.awk pyanalyzer_${vi}/project_metrics.csv | sort > api_${vi}.txt
  fi
done

echo

echo "Step 3/4: Average_changes_in_public_interfaces_between_releases"
echo ===============================================================

i=0
for (( j=1; j<$numargs; j++ )); do
  vi=${versions[$i]}
  vj=${versions[$j]}
  ${gSCRIPTS_DIR}/apidiff.sh api_${vi}.txt api_${vj}.txt > apidiff_${vi}_${vj}.txt
  res=$(tail -1 apidiff_${vi}_${vj}.txt | awk '{print $2}')
  echo Percentage_of_api_changes_between_${vi}_and_${vj} = $res
  let i++
done

echo

#
# Evolution of average cyclomatic complexity
#

echo "Step 4/4: Evolution_of_cyclomatic_complexity_between_releases"
echo =============================================================

for (( i=0; i<$numargs; i++ )); do
  vi=${versions[$i]}
  if [ "${aLANG}" == "java" ]; then
    #res=$(${gSCRIPTS_DIR}/xmlfilter.py -n -f javancss_${vi}.xml -x "/javancss/functions/function_averages/ccn/text()")
    sum_cyclo=$(awk -f $gSCRIPTS_DIR/sum_cyclomatic_complexity_java.awk javacount_$vi.csv)
    num_routines=$(cat api_$vi.txt | wc -l)
    let num_routines--
    res=$(echo "scale=2;$sum_cyclo/$num_routines" | bc)
  elif [ "${aLANG}" == "cpp" ]; then
    sum_cyclo=$(tail -1 pmccabe_${vi}.txt | awk '{print $2}')
    num_routines=$(cat api_${vi}.txt | wc -l)
    res=$(echo "scale=2;$sum_cyclo/$num_routines" | bc)
  elif [ "$aLANG" == "python" ]; then
    sum_cyclo=$(awk -f ${gSCRIPTS_DIR}/sum_cyclomatic_complexity_python.awk pyanalyzer_${vi}/project_metrics.csv)
    num_routines=$(cat api_${vi}.txt | wc -l)
    let num_routines--
    res=$(echo "scale=2;$sum_cyclo/$num_routines" | bc)
  fi
  echo Average_cyclomatic_complexity_of_version_${vi} = $res
done

echo

#
# End
#

cd $vCURDIR

source clean.sh
